<?php

/**
 * Seiten-Optionen Modul
 * Ermöglicht es einzelnen Seiten, als Kalender-Seiten zu fungieren
 */

class Trainingssystem_Plugin_Module_Seiten_Options {
    
    public function __construct() {
        add_action('add_meta_boxes', array($this, 'add_seiten_options_meta_box'));
        add_action('save_post', array($this, 'save_seiten_options'));
        add_action('wp_head', array($this, 'add_calendar_meta_tag'));
        add_action('wp_ajax_backend_seiten_options_save', array($this, 'backend_seiten_options_save'));
    }
    
    public function add_seiten_options_meta_box() {
        add_meta_box(
            'seiten_options',
            'Seiten-Optionen',
            array($this, 'seiten_options_callback'),
            'seiten',
            'side',
            'high'
        );
    }
    
    public function seiten_options_callback($post) {
        wp_nonce_field('seiten_options_nonce', 'seiten_options_nonce');
        
        $calendar_id = get_post_meta($post->ID, 'calendar_id', true);
        $is_calendar_page = !empty($calendar_id);
        
        // Prüfung: Ist das übergeordnete Training bereits als Calendar Training markiert?
        $parentTrainingIsCalendarTraining = $this->checkParentTrainingIsCalendarTraining($post->ID);
        $calendarPageDisabled = $parentTrainingIsCalendarTraining;
        $calendarPageConflictMessage = $parentTrainingIsCalendarTraining ? 
            "<div class='alert alert-warning mt-2'><small><i class='fas fa-exclamation-triangle'></i> <strong>Hinweis:</strong> Das übergeordnete Training ist bereits als Kalender-Training markiert. Ein Training kann nicht gleichzeitig als Kalender-Training verwendet werden und als Kalender-Seiten markierte Seiten enthalten.</small></div>" : 
            "";
        
        // Verfügbare Kalender laden
        $kalender = get_posts(array(
            'post_type' => 'kalender',
            'numberposts' => -1,
            'post_status' => 'publish'
        ));
        ?>
        <div class="wrapper-admin-backend">
            <div class="form-check mb-2 d-flex align-items-center">
                <input type="checkbox" 
                       class="form-check-input" 
                       id="is_calendar_page" 
                       name="is_calendar_page" 
                       value="1" 
                       <?php checked($is_calendar_page); ?>
                       <?php if($calendarPageDisabled) echo 'disabled'; ?>
                       onchange="toggleCalendarSelection()">
                <label class="ml-1 form-check-label trainings-options-label" for="is_calendar_page">Als Kalenderseite markieren</label>
            </div>
            <p class="description mb-3">Wenn aktiviert, werden alle Formulare auf dieser Seite als Kalendereinträge gespeichert.</p>
            
            <?php if($calendarPageConflictMessage): ?>
                <?php echo $calendarPageConflictMessage; ?>
            <?php endif; ?>
            
            <div id="calendar_selection_row" style="<?php echo $is_calendar_page ? '' : 'display:none;'; ?>" class="mb-3">
                <label for="calendar_id" class="form-label">Kalender auswählen:</label>
                <select id="calendar_id" name="calendar_id" class="form-control">
                    <option value="">-- Kein Kalender --</option>
                    <?php foreach($kalender as $kal): ?>
                        <option value="<?php echo $kal->ID; ?>" <?php selected($calendar_id, $kal->ID); ?>>
                            <?php echo $kal->post_title; ?>
                        </option>
                    <?php endforeach; ?>
                </select>
                <p class="description mt-1">Wählen Sie den Kalender aus, in dem die Einträge gespeichert werden sollen.</p>
            </div>

            <button id="seiten-options-save-button" class="btn btn-primary mt-3">Speichern</button>
        </div>
        
        <script>
        function toggleCalendarSelection() {
            var checkbox = document.getElementById('is_calendar_page');
            var row = document.getElementById('calendar_selection_row');
            var select = document.getElementById('calendar_id');
            
            // Nur ausführen wenn Checkbox nicht deaktiviert ist
            if(checkbox.disabled) {
                return;
            }
            
            if(checkbox.checked) {
                row.style.display = '';
            } else {
                row.style.display = 'none';
                select.value = '';
            }
        }
        </script>
        <?php
    }
    
    public function save_seiten_options($post_id) {
        if (!isset($_POST['seiten_options_nonce']) || 
            !wp_verify_nonce($_POST['seiten_options_nonce'], 'seiten_options_nonce')) {
            return;
        }
        
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }
        
        if (!current_user_can('edit_post', $post_id)) {
            return;
        }
        
        $calendar_id = isset($_POST['calendar_id']) ? sanitize_text_field($_POST['calendar_id']) : '';
        update_post_meta($post_id, 'calendar_id', $calendar_id);
    }
    
    public function add_calendar_meta_tag() {
        // Nur auf Seiten des Typs 'seiten' ausführen
        if (is_singular('seiten')) {
            $calendar_id = get_post_meta(get_the_ID(), 'calendar_id', true);
            if (!empty($calendar_id)) {
                echo '<meta name="calendar_id" content="' . esc_attr($calendar_id) . '">' . "\n";
                
            }
        }
    }
    
    public function backend_seiten_options_save() {
        if (isset($_POST['seitenid']) && isset($_POST['is_calendar_page']) && isset($_POST['calendar_id'])) {
            
            $seitenid = intval(sanitize_text_field($_POST['seitenid']));
            $is_calendar_page = strval(sanitize_text_field($_POST['is_calendar_page']));
            $calendar_id = strval(sanitize_text_field($_POST['calendar_id']));
            
            // Kalender-ID nur speichern, wenn als Kalenderseite markiert
            if ($is_calendar_page === '1') {
                update_post_meta($seitenid, 'calendar_id', $calendar_id);
            } else {
                delete_post_meta($seitenid, 'calendar_id');
            }
            
            echo '1';
        } else {
            echo '0';
        }
        wp_die();
    }
    
    /**
     * Prüft ob das übergeordnete Training bereits als Calendar Training markiert ist
     * 
     * @param int $seitenId Seiten ID
     * @return bool True wenn das übergeordnete Training als Calendar Training markiert ist
     */
    private function checkParentTrainingIsCalendarTraining($seitenId) {
        global $wpdb;
        
        // Über die Trainingseiten-Tabelle das Training finden
        $seitenTable = $wpdb->prefix . TRAININGSSYSTEM_PLUGIN_DB_TRAINING_SEITEN;
        $seitenRow = $wpdb->get_row($wpdb->prepare(
            "SELECT training_id FROM $seitenTable WHERE seiten_id = %d",
            $seitenId
        ));
        
        if (!$seitenRow) {
            return false;
        }
        
        $trainingId = $seitenRow->training_id;
        
        // Prüfen ob das Training als Calendar Training markiert ist
        $calendarTraining = get_post_meta($trainingId, 'calendarTraining', true);
        return ($calendarTraining === "1");
    }
    
}

// Initialisieren
new Trainingssystem_Plugin_Module_Seiten_Options();
